﻿using System;
using System.Collections.Generic;
using System.Linq;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

using HIPS.Web.Components.ServiceModel;

using ModelInterface = HIPS.Web.ModelInterface.Common;
using Model = HIPS.Web.Model.Common;

using Moq;

namespace HIPS.Web.Test.Common
{

    /// <summary>
    /// Provides general methods to assist with unit testing.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
    /// </history>
    internal class TestAssistant
    {

        #region Declarations

        internal const string HOSPITAL_CODE_SYSTEM = "pasFacCd";

        #region Mock Repositories

        /// <summary>
        /// Provides access to mock repositories.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
        /// </history>
        internal class MockRepository
        {

            #region Methods

            /// <summary>
            /// Gets a mock hospital repository with a specified number of mock hospitals.
            /// </summary>
            /// <param name="numberOfHospitals">Number of hospitals to provide through the mock repository.</param>
            /// <returns>Mock IHospitalRepository.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
            /// </history>
            internal static Mock<ModelInterface.Common.IHospitalRepository> GetHospitalRepository(int numberOfHospitals)
            {
                var mock = new Mock<ModelInterface.Common.IHospitalRepository>();

                //IHospitalRepository.GetHospitals:
                mock.Setup(m => m.GetHospitals(It.IsAny<string>()))
                    .Returns((string s) =>
                        Enumerable.Range(1, numberOfHospitals)
                            .Select(h => new Hospital() { Name = string.Format("Hospital {0}", h), Codes = new List<HospitalCode>() { new HospitalCode() { CodeSystemCode = s, Code = string.Format("H{0}", h) } } }).ToList()
                    );

                return mock;
            }

            /// <summary>
            /// Gets a mock settings repository.
            /// </summary>
            /// <returns>Mock ISettingsRepositry.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
            /// </history>
            internal static Mock<ModelInterface.Common.ISettingsRepository> GetSettingsRepository()
            {
                var mock = new Mock<ModelInterface.Common.ISettingsRepository>();

                //ISettingsRepository.GetSettings:
                mock.Setup(m => m.GetSettings())
                    .Returns(
                        new List<Model.Common.Setting>()
                        {
                            new Model.Common.Setting() { Code = Model.Common.Setting.SettingCodes.DefaultHospitalCodeSystem.ToString(), Value = "pasFacCd" },
                            new Model.Common.Setting() { Code = Model.Common.Setting.SettingCodes.PcehrViewPDDocumentClasses.ToString(), Value = "100.16765,100.16764" },
                            new Model.Common.Setting() { Code = Model.Common.Setting.SettingCodes.PcehrViewServiceDateDocumentClasses.ToString(), Value = "18842-5" },
                            new Model.Common.Setting() { Code = Model.Common.Setting.SettingCodes.PcehrViewFromDateOffsetMonths.ToString(), Value = "-24" },
                            new Model.Common.Setting() { Code = Model.Common.Setting.SettingCodes.PcehrViewPatientDaysDischarged.ToString(), Value = "5" }
                        }
                    );

                return mock;
            }

            /// <summary>
            /// Gets a mock patient repository for a specified hospital and with mock patients.
            /// </summary>
            /// <param name="hospital">Hospital the patient repository represents.</param>
            /// <param name="numberOfPatientsWithPcehr">Number of patients with a PCEHR to provide through the mock repository.</param>
            /// <param name="numberOfPatientsWithoutPcehr">Number of patients without a PCEHR to provide through the mock repository.</param>
            /// <returns>Mock IPatientRepository.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
            /// </history>
            internal static Mock<ModelInterface.Common.IPatientRepository> GetPatientRepository(Hospital hospital, int numberOfPatientsWithPcehr, int numberOfPatientsWithoutPcehr)
            {
                var mock = new Mock<ModelInterface.Common.IPatientRepository>();

                //IPatientRepository.ListPatientsCurrentlyInHospital:
                mock.Setup(m => m.ListPatientsCurrentlyInHospital(It.IsAny<string>(), It.IsAny<string>(), It.IsAny<bool?>(), It.IsAny<UserDetails>()))
                    .Returns((string hospitalCodeSystem, string hospitalCode, bool? withPcehr, UserDetails operatingUser) =>
                        {
                            List<PatientSchemas.PatientInHospital> patients = new List<PatientSchemas.PatientInHospital>();

                            //Generate patients with PCEHR if required.
                            if ((numberOfPatientsWithPcehr > 0) && withPcehr != false)
                            {
                                patients.AddRange(
                                    Enumerable.Range(1, numberOfPatientsWithPcehr)
                                        .Select(p => new PatientSchemas.PatientInHospital()
                                        {
                                            ParticipationStatus = ConsentSchemas.ParticipationStatus.PcehrAdvertised,
                                            AdmissionDate = DateTime.Now.AddDays(-p * 2),
                                            Bed = string.Format("Bed {0}", p),
                                            DateOfBirth = DateTime.Now.AddYears(-p),
                                            DischargeDate = null,
                                            FamilyName = string.Format("Family {0}+", p),
                                            GivenNames = string.Format("Given {0}", p),
                                            HospitalCode = hospital.Codes.FirstOrDefault(c => c.CodeSystemCode == HOSPITAL_CODE_SYSTEM).Code, 
                                            HospitalName = hospital.Name,
                                            Mrn = string.Format("{0}+", p), 
                                            Room = string.Format("Room {0}", p), 
                                            Ward = string.Format("Ward {0}", p)
                                        } )
                                    );
                            }

                            //Generate patients without PCEHR if required.
                            if ((numberOfPatientsWithoutPcehr > 0) && (withPcehr != true))
                            {
                                patients.AddRange(
                                    Enumerable.Range(1, numberOfPatientsWithoutPcehr)
                                        .Select(p => new PatientSchemas.PatientInHospital()
                                        {
                                            ParticipationStatus = ConsentSchemas.ParticipationStatus.PcehrNotAdvertised,
                                            AdmissionDate = DateTime.Now.AddDays(-p * 2),
                                            Bed = string.Format("Bed {0}", p),
                                            DateOfBirth = DateTime.Now.AddYears(-p),
                                            DischargeDate = null,
                                            FamilyName = string.Format("Family {0}-", p),
                                            GivenNames = string.Format("Given {0}", p),
                                            HospitalCode = hospital.Codes.FirstOrDefault(c => c.CodeSystemCode == HOSPITAL_CODE_SYSTEM).Code,
                                            HospitalName = hospital.Name,
                                            Mrn = string.Format("{0}-", p), 
                                            Room = string.Format("Room {0}", p), 
                                            Ward = string.Format("Ward {0}", p)
                                        } )
                                    );
                            }

                            return new ServiceResponse<PatientSchemas.PatientListResponse>
                            (
                                data: new PatientSchemas.PatientListResponse()
                                {
                                    HipsResponse = new HipsResponse(HipsResponseIndicator.OK),
                                    PatientInHospitalList = patients
                                },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                            );

                        }
                    );

                return mock;
            }

            /// <summary>
            /// Gets a mock patient repository for a specified hospital and with mock patients.
            /// </summary>
            /// <param name="hospital">Hospital the patient repository represents.</param>
            /// <param name="numberOfPatientsWithPcehr">Number of patients with a PCEHR to provide through the mock repository.</param>
            /// <param name="numberOfPatientsWithoutPcehr">Number of patients without a PCEHR to provide through the mock repository.</param>
            /// <returns>Mock IPatientRepository.</returns>
            /// <summary>
            /// Gets a mock PCEHR view repository for a specified hospital and patient.
            /// </summary>
            /// <param name="hospital">Hospital the patient is resident in.</param>
            /// <param name="patient">Patient the repository will be used with.</param>
            /// <param name="documentClasses">A list of 2-item tuples containing the document class code and description for document classes generated by the repository.</param>
            /// <param name="numberOfDocumentsPerClass">Number of documents to be generated for each document class returned by the repository.</param>
            /// <param name="currentAccessLevel">Access level to the PCEHR for patients / documents. Defaults to "WithoutCode".</param>
            /// <param name="allowUnknownAccessWithoutCode">For an Access Level of "Unknown", indicates whether access should be allowed without a code. Defaults to true.</param>
            /// <returns>Mock IPcehrViewRepository.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="08 December 2013">Initial version.</change>
            /// </history>
            internal static Mock<ModelInterface.PcehrView.IPcehrViewRepository> GetPcehrViewRepository(Hospital hospital, PatientSchemas.PatientInHospital patient, List<Tuple<string, string>> documentClasses, int numberOfDocumentsPerClass, AccessCodeRequired currentAccessLevel = PcehrDataStore.Schemas.Enumerators.AccessCodeRequired.WithoutCode, bool allowUnknownAccessWithoutCode = true)
            {
                var mock = new Mock<ModelInterface.PcehrView.IPcehrViewRepository>();

                //IPcehrViewRepository.ListActiveDocuments:
                mock.Setup(m => m.ListActiveDocuments(It.IsAny<Mrn>(), It.IsAny<UserDetails>()))
                    .Returns((Mrn patientIdentifier, UserDetails operatingUser) =>
                        {
                            List<DocumentMetaDataItem> documents = new List<DocumentMetaDataItem>();

                            foreach (var currentDocumentClass in documentClasses)
                            {
                                //Generate documents.
                                if (numberOfDocumentsPerClass > 0)
                                {
                                    documents.AddRange(
                                        Enumerable.Range(1, numberOfDocumentsPerClass)
                                            .Select(d => new DocumentMetaDataItem()
                                            {
                                                AuthorInstitutionName = hospital.Name,
                                                AuthorPerson = string.Format("Author {0}", d),
                                                AuthorPersonFamilyName = string.Format("Author {0} Family", d),
                                                AuthorPersonGivenName = string.Format("Author {0} Given", d),
                                                AuthorPersonNamePrefix = string.Format("Author {0} Prefix", d),
                                                CreationTime = DateTime.Now.AddDays(10),
                                                DocumentClassCode = currentDocumentClass.Item1,
                                                DocumentClassName = currentDocumentClass.Item2,
                                                DocumentUniqueId = d.ToString(),
                                                RepositoryUniqueId = currentDocumentClass.Item1,
                                                ServiceStartTime = DateTime.Now.AddDays(-d * 2),
                                                ServiceStopTime = DateTime.Now.AddDays(-d)
                                            })
                                        );
                                }
                            }

                            return new ServiceResponse<DocumentListResponse<PatientIdentifierBase>>
                            (
                                data: new DocumentListResponse<PatientIdentifierBase>()
                                {
                                    HipsResponse = new HipsResponse(HipsResponseIndicator.OK),
                                    PatientIdentifier = patientIdentifier,
                                    DocumentList = documents
                                },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                            );
                        }
                    );

                //IPcehrViewRepository.GetDocument:
                mock.Setup(m => m.GetDocument(It.IsAny<Mrn>(), It.IsAny<UserDetails>(), It.IsAny<DocumentRequest>()))
                    .Returns((Mrn patientIdentifier, UserDetails operatingUser, DocumentRequest documentSpec) =>
                        {
                            return new ServiceResponse<DocumentResponse>
                            (
                                data: new DocumentResponse()
                                {
                                    Attachments = new List<Attachment>() { new Attachment() { Contents = HIPS.Web.Test.Properties.Resources.logo, FileName = "logo.png" } },
                                    Document = HIPS.Web.Test.Properties.Resources.CDA_ROOT,
                                    DocumentUniqueId = documentSpec.DocumentUniqueId,
                                    FileName = "CDA_ROOT.xml",
                                    HipsResponse = new HipsResponse(HipsResponseIndicator.OK),
                                    MimeType = "text/xml",
                                    RepositoryUniqueId = documentSpec.RepositoryUniqueId
                                },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                            );
                        }
                    );

                //TODO: Create mock method.
                //IPcehrViewRepository.GetView:
                mock.Setup(m => m.GetView(It.IsAny<Mrn>(), It.IsAny<UserDetails>(), It.IsAny<PrescriptionAndDispenseViewRequest>()));

                //IPechrViewRepository.IsPcehrAdvertised:
                mock.Setup(m => m.IsPcehrAdvertised(It.IsAny<Mrn>(), It.IsAny<DateTime>(), It.IsAny<UserDetails>()))
                    .Returns(() =>
                        {
                            return new ServiceResponse<DoesPcehrExistResponse>
                            (
                                data: new DoesPcehrExistResponse()
                                {
                                    AccessCodeRequired = currentAccessLevel,
                                    HipsResponse = new HipsResponse(HipsResponseIndicator.OK),
                                    PcehrExists = true
                                },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                           );
                        }
                    );

                //IPcehrViewRepository.GainAccessWithoutCode:
                mock.Setup(m => m.GainAccessWithoutCode(It.IsAny<Mrn>(), It.IsAny<UserDetails>()))
                    .Returns(() =>
                    {
                        if ((currentAccessLevel == AccessCodeRequired.WithoutCode) || (currentAccessLevel == AccessCodeRequired.Unknown && allowUnknownAccessWithoutCode))
                        {
                            return new ServiceResponse<GainPcehrAccessResponse>
                            (
                                data: new GainPcehrAccessResponse() { AccessPermission = GainPcehrAccessStatus.Permit, HipsResponse = new HipsResponse(HipsResponseIndicator.OK) },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                            );
                        }
                        else
                        {
                            return new ServiceResponse<GainPcehrAccessResponse>
                            (
                                data: new GainPcehrAccessResponse() { AccessPermission = GainPcehrAccessStatus.Deny, HipsResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError) },
                                isSuccessful: false,
                                messages: new ResponseMessageList()
                            );
                        }
                    }                    
                    );

                //IPcehrViewRepository.GainAccessWithCode (RAC, LDAC):
                mock.Setup(m => m.GainAccessWithCode(It.IsAny<Mrn>(), It.IsIn<string>("RAC", "LDAC"), It.IsAny<UserDetails>()))
                    .Returns((Mrn patientIdentifier, string accessCode, UserDetails operatingUser) =>
                        {
                            return new ServiceResponse<GainPcehrAccessResponse>
                            (
                                data: new GainPcehrAccessResponse()
                                {
                                    AccessPermission = GainPcehrAccessStatus.Permit,
                                    HipsResponse = new HipsResponse(HipsResponseIndicator.OK)
                                },
                                isSuccessful: true,
                                messages: new ResponseMessageList()
                            );
                        }
                    );

                //IPcehrViewRepository.GainAccessWithCode (INVALID):
                mock.Setup(m => m.GainAccessWithCode(It.IsAny<Mrn>(), It.IsNotIn<string>("RAC", "LDAC"), It.IsAny<UserDetails>()))
                    .Returns((Mrn patientIdentifier, string accessCode, UserDetails operatingUser) =>
                    {
                        return new ServiceResponse<GainPcehrAccessResponse>
                        (
                            data: new GainPcehrAccessResponse()
                            {
                                AccessPermission = GainPcehrAccessStatus.Deny,
                                HipsResponse = new HipsResponse(HipsResponseIndicator.PcehrServiceError) { ResponseCode = "PCEHR_ERROR_5103", ResponseCodeDescription = "eHealth Record is found but Access Code is invalid" },
                            },
                            isSuccessful: false,
                            messages: new ResponseMessageList() { new ResponseMessage("PCEHR_ERROR_5103 - eHealth Record is found but Access Code is invalid", Components.Common.MessageLevel.Warning) }
                        );
                    }
                    );

                //IPcehrViewRepository.GainAccessEmergency:
                mock.Setup(m => m.GainAccessEmergency(It.IsAny<Mrn>(), It.IsAny<UserDetails>()))
                    .Returns((Mrn patientIdentifier, UserDetails operatingUser) =>
                    {
                        return new ServiceResponse<GainPcehrAccessResponse>
                        (
                            data: new GainPcehrAccessResponse()
                            {
                                AccessPermission = GainPcehrAccessStatus.Permit,
                                HipsResponse = new HipsResponse(HipsResponseIndicator.OK)
                            },
                            isSuccessful: true,
                            messages: new ResponseMessageList()
                        );
                    }
                    );

                return mock;
            }

            #endregion

        }

        #endregion

        #endregion

        #region Methods

        /// <summary>
        /// Returns a patient identifier for a known patient.
        /// </summary>
        /// <param name="patient">A value representing the patient.</param>
        /// <returns>The patient identifier.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        internal static Mrn GetKnownPatientIdentifier(KnownPatients patient)
        {
            Mrn result;

            switch (patient)
            {
                case (KnownPatients.BernardLam):
                    {
                        result = new Mrn("TEST-IHI_4", "RCH", HOSPITAL_CODE_SYSTEM);
                        break;
                    }
                case (KnownPatients.JakeBerman):
                    {
                        result = new Mrn("TEST-NOC_17", "RCH", HOSPITAL_CODE_SYSTEM);
                        break;
                    }
                case (KnownPatients.FrederickForde):
                    {
                        result = new Mrn("TEST-CCA_34", "RCH", HOSPITAL_CODE_SYSTEM);
                        break;
                    }
                case (KnownPatients.DarrylBarker):
                    {
                        result = new Mrn("TEST-SAH_5", "RKH", HOSPITAL_CODE_SYSTEM);
                        break;
                    }
                case (KnownPatients.PhilipLudwigHobbs):
                    {
                        result = new Mrn("RENDERING", "RCH", HOSPITAL_CODE_SYSTEM);
                        break;
                    }
                default:
                    {
                        result = null;
                        break;
                    }
            }

            return result;
        }

        /// <summary>
        /// Returns the current user details.
        /// </summary>
        /// <returns>Current user details.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        internal static UserDetails GetCurrentUserDetails()
        {
            UserDetails result = new UserDetails();

            System.Security.Principal.IPrincipal p = System.Threading.Thread.CurrentPrincipal;

            if (p != null)
            {
                result.AuthorisedEmployeeUserId = p.Identity.Name;
                result.Domain = "-";
                result.Login = p.Identity.Name;
                result.Name = p.Identity.Name;
                result.Role = UserRole.InteractiveUser;
            }

            return result;
        }

        /// <summary>
        /// Gets a list of tuples containing document classes used in unit tests.
        /// </summary>
        /// <returns>List of document classes.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="08 December 2013">Initial version.</change>
        /// </history>
        internal static List<Tuple<string, string>> GetDocumentClasses()
        {
            return new List<Tuple<string, string>>()
            {
                new Tuple<string, string>("18842-5", "Discharge Summary"),
                new Tuple<string, string>("100.16765", "e-Prescription"),
                new Tuple<string, string>("100.16764", "Dispense Record"),
                new Tuple<string, string>("", "Other")
            };
        }

        #endregion

    }
}
